// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#include "meminterface.h"
#include "registers.h"
#include "ini.h"
#include "interrupt.h"

const char *bat_size_string(DWORD BL) {
	switch(BL) {
	case 0x000:
		return "128KB";
	case 0x001:
		return "256KB";
	case 0x003:
		return "512MB";
	case 0x007:
		return "1MB";
	case 0x00F:
		return "2MB";
	case 0x01F:
		return "4MB";
	case 0x03F:
		return "8MB";
	case 0x07F:
		return "16MB";
	case 0x0FF:
		return "32MB";
	case 0x1FF:
		return "64MB";
	case 0x3FF:
		return "128MB";
	case 0x7FF:
		return "256MB";
	default:
		return "invalid";
	}
}
string bat_wimg_string(DWORD bat) {
	ostringstream str;
	str << (getbit(bat, 25) ? "W" : "") << (getbit(bat, 26) ? "I" : "") <<
		(getbit(bat, 27) ? "M" : "") << (getbit(bat, 28) ? "G" : "");
	return str.str();
}
const char *bat_pp_string(DWORD bat) {
	if(getbit(bat, 31))
		return "readonly";
	else if(getbit(bat, 30))
		return "read/write";
	else
		return "no access";
}

bool dbat_is_valid(DWORD batu, DWORD batl) {
	//If no Valid bits are set, the BAT is ignored.
	if(!getbit(batu, 30) && !getbit(batu, 31))
		return true;

	int first_one = 30;
	//BL must be a straight bitmask
	bool one=false;
	for(int i=19; i<=29; i++) {
		if(one && !getbit(batu, i)) {
			DEGUB("BL is not a straight bitmask\n");
			return false;
		}
		if(getbit(batu, i)) {
			one = true;
			if(i < first_one)
				first_one = i;
		}
	}

	//Bits (first_one - 15) to 24 must be 0 //no, they are just reserved/ignored
	/*if(getbitsw(batl, (first_one - 15), 24) != 0) {
	DEGUB("Bits (first_one(%i) - 15)(%i) to 24 are not 0\n", first_one, (first_one - 15));
	return false;
	}*/
	//Bit 29 must be 0
	if(getbit(batl, 29)) {
		DEGUB("Bit 29 is not 0\n");
		return false;
	}

	return true;
}
bool ibat_is_valid(DWORD batu, DWORD batl) {
	//If no Valid bits are set, the BAT is ignored.
	if(!getbit(batu, 30) && !getbit(batu, 31))
		return true;
	if(!dbat_is_valid(batu, batl))
		return false;
	//W or G bits may not be set
	if(getbit(batl, 25) || getbit(batl, 28)) {
		DEGUB("W(25) or G(28) bits may not be set\n");
		return false;
	}

	return true;
}

MemInterface::MemInterface(RegistersBase &_r, InterruptRaiser &i, Hardware &_h,
													 bool rec) :
r(_r), interrupt(i), ibats_valid(true), dbats_valid(true), dpage_faults(0),
ipage_faults(0), recompiler(rec), reg(_r), locked_cache(LOCKED_CACHE_SIZE),
mainmem(MAIN_MEMORY_SIZE), cachemem(CACHE_SIZE), h(_h), efb(EFB_SIZE) {
	memset(bat_enabled, 0, sizeof(bat_enabled));
	//memset(bat_valid, 1, sizeof(bat_valid));
	memset(mainmem, 0, mainmem.size());
	memset(cachemem, 0, cachemem.size());
	memset(locked_cache, 0, locked_cache.size());
	clear_degub();
}

MemInterface::~MemInterface() {
	if(g::advanced_mode) {
		if(!this->recompiler) {
			DEGUB("%i Instruction Page Faults ", ipage_faults);
		}
		DEGUB("%i Data Page Faults.\n", dpage_faults);
	}
}

DWORD MemInterface::getmsr() const {
	return r.getmsr();
}

void MemInterface::msr_notify() {
	/*if(msrbits != (r.getmsr() & (MSR_PR | MSR_IR | MSR_DR))) {
	DEGUB("MSR bits:%s%s%s\n", (r.getmsr() & MSR_PR) ? " PR" : "",
	(r.getmsr() & MSR_IR) ? " IR" : "", (r.getmsr() & MSR_DR) ? " DR" : "");
	msrbits = r.getmsr() & (MSR_PR | MSR_IR | MSR_DR);
	}*/
	if(!ibats_valid && (r.getmsr() & MSR_IR)) {
		DEGUB("One or more IBATs are invalid. No instructions may be fetched... Instant BFE!!!\n");
		throw generic_fatal_exception("Invalid IBAT");
	}
	if(!dbats_valid && (r.getmsr() & MSR_DR)) {
		DEGUB("One or more DBATs are invalid. No data may be fetched... Instant BFE!!!\n");
		throw generic_fatal_exception("Invalid DBAT");
	}
}

void MemInterface::bat_notify(DWORD spr) {
	const char *pointer;
	bool temp_be;
	{
		int i = getbitsw(spr, 29, 30);
		DWORD batu = getbit(spr, 28) ? r.dbatu[i] : r.ibatu[i];
		temp_be = getbitsw(batu, 30, 31) != 0;
	}

	if(!bat_enabled[getbitsw(spr, 28, 30)] && !temp_be)
		pointer = "ignored";
	else if(bat_enabled[getbitsw(spr, 28, 30)] && !temp_be)
		pointer = "disabled";
	else if(!bat_enabled[getbitsw(spr, 28, 30)] && temp_be)
		pointer = "enabled";
	else if(bat_enabled[getbitsw(spr, 28, 30)] && temp_be)
		pointer = "updated";
	else
		throw generic_fatal_exception("Bad BAT notify!");

	if(bat_enabled[getbitsw(spr, 28, 30)] || temp_be) {
		DEGUB("%cBAT%i %s\n",
			getbit(spr, 28) ? 'D' : 'I', getbitsw(spr, 29, 30), pointer);
	}
	if(temp_be) {
		//for(int i=(getbit(spr, 28)?536:528); i<(getbit(spr, 28)?544:536); i+=2)
		dump_bat(spr);
	}
	bat_enabled[getbitsw(spr, 28, 30)] = temp_be;

	//if(g::advanced_mode && (r.getmsr() & (getbit(spr, 28) ? MSR_DR : MSR_IR)))
	//throw generic_fatal_exception("BAT change while translation enabled!");

	if(bat_enabled[getbitsw(spr, 28, 30)] || temp_be)
		rebuild_batv(getbit(spr, 28));  //This operation is assumed to be rare.

}

void MemInterface::bat_notify_all() {
	for(int i=0; i<4; i++) {
		bat_enabled[i] = getbitsw(r.ibatu[i], 30, 31) != 0;
		bat_enabled[4+i] = getbitsw(r.dbatu[i], 30, 31) != 0;
	}
	rebuild_batv(true);
	rebuild_batv(false);
}

void MemInterface::rebuild_batv(bool data) {
	if(g::advanced_mode) {
#define DI_FIRST(basename) (data ? (d##basename) : (i##basename))
		std::vector<BAT> &batv = DI_FIRST(batv);
		bool &bats_valid = DI_FIRST(bats_valid);
		bool (*bat_is_valid)(DWORD, DWORD) = DI_FIRST(bat_is_valid);
		DWORD *batu = data ? r.dbatu : r.ibatu;
		DWORD *batl = data ? r.dbatl : r.ibatl;

		batv.clear();
		bats_valid = true;
		for(int i=0; i<4; i++) {
			if(!bat_is_valid(batu[i], batl[i])) {
				DEGUB("%cBAT%i invalid\n", data ? 'D' : 'I', i);
				bats_valid = false;
			} else if(getbitsw(batu[i], 30, 31) != 0) {
				BAT bat;
				bat.mask = ~(getbitsw(batu[i], 19, 29) << 17 | 0x1FFFF);
				bat.src = batu[i] & bat.mask;
				bat.dst = batl[i] & bat.mask;
				bat.supervisor = getbit(batu[i], 30);
				bat.user = getbit(batu[i], 31);
				bat.read = getbit(batl[i], 30) || getbit(batl[i], 31);
				bat.write = getbit(batl[i], 30) && !getbit(batl[i], 31);

				//check for overlaps
				for(size_t j=0; j<batv.size(); j++) {
					BAT bat2 = batv[j];
					if((bat.src >= bat2.src && bat.src <= bat2.src + ~bat2.mask) ||
						(bat.src + ~bat.mask >= bat2.src &&
						bat.src + ~bat.mask <= bat2.src + ~bat2.mask))
					{
						throw generic_fatal_exception("BAT overlap!");
					}
				}

				batv.push_back(bat);
			}
		}
	}
}

DWORD MemInterface::translate_i(DWORD ea) {
	if(!g::advanced_mode || !(r.getmsr() & MSR_IR))
		return ea;

	for(size_t i=0; i<ibatv.size(); i++) {
		BAT bat = ibatv[i];
		if((ea & bat.mask) == bat.src) {
			if((r.getmsr() & MSR_PR) ? bat.user : bat.supervisor) {
				if(bat.read) {
					return ea & ~bat.mask | bat.dst;
				} else {
					DEGUB("Memory protection violation! ISI EA=0x%08X\n", ea);
					//interrupt.raise(INTERRUPT_ISI);
					throw generic_fatal_exception("Memory protection violation! ISI");
				}
			}
		}
	}

	return page_table_search(ea, true, false, recompiler);
}

DWORD MemInterface::translate_d_read(DWORD ea) {
	return translate_d(ea, false, false);
}
DWORD MemInterface::translate_d_write(DWORD ea) {
	return translate_d(ea, true, false);
}
DWORD MemInterface::translate_d_read_rec(DWORD ea) {
	return translate_d(ea, false, true);
}
DWORD MemInterface::translate_d_write_rec(DWORD ea) {
	return translate_d(ea, true, true);
}
DWORD MemInterface::translate_d(DWORD ea, bool write, bool recompiler) {
	MYASSERT(g::advanced_mode);
	if(!(r.getmsr() & MSR_DR))
		return ea;

	for(size_t i=0; i<dbatv.size(); i++) {
		BAT bat = dbatv[i];
		if((ea & bat.mask) == bat.src) {
			if((r.getmsr() & MSR_PR) ? bat.user : bat.supervisor) {
				if(write ? bat.write : bat.read) {
					return ea & ~bat.mask | bat.dst;
				} else {
					DEGUB("Memory protection violation! DSI %s EA=0x%08X\n",
						write ? "write" : "read", ea);
					//interrupt.raise(INTERRUPT_DSI);
					throw generic_fatal_exception("Memory protection violation! DSI");
				}
			}
		}
	}

	return page_table_search(ea, false, write, recompiler);
}

DWORD MemInterface::page_table_search(DWORD ea, bool instruction, bool write,
																			bool recompiler)
{
	MYASSERT(!(instruction && write));

	//DEGUB("%s%s EA 0x%08X\n", instruction ? "Instruction" : "",
	//write ? "Write to" : "Read from", ea);
	DWORD segnum = getbitsw(ea, 0, 3);
	if(getbit(r.sr[segnum], 0))
		throw generic_fatal_exception("Direct-Store segment encountered!");
	if(instruction && getbit(r.sr[segnum], 3))
		throw generic_fatal_exception("No-execute segment encountered!");
	//QWORD va = getbitsw(ea, 4, 31) |
	//(QWORD(getbitsw(r.sr[segnum], 8, 31)) << (16 + 12));
	//DEGUB("Virtual address: 0x%013I64X\n", va);
	//yadda yadda
	DWORD hash1, hash2, pta1, pta2;
	hash1 = getbitsw(r.sr[segnum], 32-19, 31) ^ (getbitsw(ea, 4, 19));
	hash2 = ~hash1;
	pta1 = (r.sdr1 & makemaskw(0, 6)) |
		((getbitsw(r.sdr1, 7, 15) | ((hash1 >> 10) & getbitsw(r.sdr1, 23, 31))) << 16) |
		((hash1 & 0x3FF) << 6);
	pta2 = (r.sdr1 & makemaskw(0, 6)) |
		((getbitsw(r.sdr1, 7, 15) | ((hash2 >> 10) & getbitsw(r.sdr1, 23, 31))) << 16) |
		((hash2 & 0x3FF) << 6);
	//DEGUB("PTA 1: 0x%08X. PTA 2: 0x%08X.\n", pta1, pta2);

	DWORD pa = page_group_search(ea, instruction, write, false, pta1, r.sr[segnum]);
	if(pa == BAD_ADDRESS)
		pa = page_group_search(ea, instruction, write, true, pta2, r.sr[segnum]);
	if(pa != BAD_ADDRESS) {
		return pa;
	}

	if(instruction) {
		if(recompiler) {
			return BAD_ADDRESS;
		} else {  //interpreter
			//if(g::log_interrupts) {
			DEGUB("iPage fault: 0x%08X\n", ea);
			//}
			raiseISI();
		}
	} else {  //data
		//if(g::log_interrupts || g::verbose) {
		DEGUB("dPage fault: 0x%08X\n", ea);
		//}
		dpage_faults++;
		r.dar = ea;
		r.dsisr = makeflag(1);
		interrupt.raise(INTERRUPT_DSI);
	}

	if(recompiler)
		return BAD_ADDRESS;
	else
		throw page_fault_exception("Page fault!");
}

void MemInterface::raiseISI() {
	ipage_faults++;
	r.srr1 &= ~makemaskw(1, 4);
	r.srr1 |= makeflag(1);
	interrupt.raise(INTERRUPT_ISI);
}

DWORD MemInterface::page_group_search(DWORD ea, bool instruction, bool write,
																			bool secondary, DWORD pta, DWORD sr)
{
	//Scan the PTEGs till we find a match.
	//BYTE *ptp = getp_physical(pta, 64);
	for(int i=0; i<8; i++) {
		//DWORD pteh = swapw(*MAKEP(DWORD, ptp));
		DWORD pteh = prw(pta);
		//DEGUB("%cPTE%i: %08X %08X\n", primary ? 'P' : 'S', i, pteh,
		//swapw(*MAKEP(DWORD, ptp + 4))); //Don't forget to swap!
		if((getbit(pteh, 25) == secondary) && getbit(pteh, 0) &&
			getbitsw(pteh, 1, 24) == getbitsw(sr, 8, 31) &&
			getbitsw(pteh, 26, 31) == getbitsw(ea, 4, 9))
		{
			//DWORD ptel = swapw(*MAKEP(DWORD, ptp + 4));
			DWORD ptel = prw(pta + 4);
			//DEGUB("Match found in %sary PTEG. PTE%i: %08X %08X\n",
			//primary ? "prim" : "second", i, pteh, ptel);
			if(!getbit(ptel, 23)) {
				ptel |= makeflag(23);
				//*MAKEP(DWORD, ptp + 4) = swapw(ptel);
				pww(pta + 4, ptel);
				DEGUB("R flag set.\n");
			}

			bool key = getbit(sr, (r.getmsr() & MSR_PR) ? 2 : 1);
			DWORD pp = getbitsw(ptel, 30, 31);
			bool prohibited;
			if(write) {
				prohibited = pp == 3 || (key && !(pp & 2));
			} else {	//read
				prohibited = key && !pp;
			}
			if(prohibited) {
				throw generic_fatal_exception(instruction ? "I" : "D" +
					std::string("SI: Protection violation"));
			} else {
				if(write && !getbit(ptel, 24)) {
					ptel |= makeflag(24);
					//*MAKEP(DWORD, ptp + 4) = swapw(ptel);
					pww(pta + 4, ptel);
					DEGUB("C flag set.\n");
				}
				DWORD pa = (ptel & 0xFFFFF000) | (ea & 0xFFF);
				//DEGUB("Translated address: 0x%08X\n", pa);
				return pa;
			}
		}
		pta += 8;
	}

	return BAD_ADDRESS;
}

void MemInterface::dump_bat(DWORD spr) {
	int i = getbitsw(spr, 29, 30);
	if(!getbit(spr, 28)) {  //IBAT
		DWORD BL = getbitsw(r.ibatu[i], 19, 29);
		DEGUB("IBAT%iU: %08X (BEPI: %04X, Effective BL mask: %08X, Size: %s, Vs:%i, Vp:%i\n",
			i, r.ibatu[i], getbitsw(r.ibatu[i], 0, 14) << 1,
			BL << 17 | 0x1FFFF, bat_size_string(BL), getbit(r.ibatu[i], 30), 
			getbit(r.ibatu[i], 31));
		DEGUB("IBAT%iL: %08X (BPRN: %04X, WIMG: %s, Protection: %s) Valid: %s\n",
			i, r.ibatl[i], getbitsw(r.ibatl[i], 0, 14) << 1,
			bat_wimg_string(r.ibatl[i]).c_str(), bat_pp_string(r.ibatl[i]),
			ibat_is_valid(r.ibatu[i], r.ibatl[i]) ? "yes" : "no");
	} else {  //DBAT
		DWORD BL = getbitsw(r.dbatu[i], 19, 29);
		DEGUB("DBAT%iU: %08X (BEPI: %04X, Effective BL mask: %08X, Size: %s, Vs:%i, Vp:%i\n",
			i, r.dbatu[i], getbitsw(r.dbatu[i], 0, 14) << 1,
			BL << 17 | 0x1FFFF, bat_size_string(BL), getbit(r.dbatu[i], 30), 
			getbit(r.dbatu[i], 31));
		DEGUB("DBAT%iL: %08X (BPRN: %04X, WIMG: %s, Protection: %s) Valid: %s\n",
			i, r.dbatl[i], getbitsw(r.dbatl[i], 0, 14) << 1,
			bat_wimg_string(r.dbatl[i]).c_str(), bat_pp_string(r.dbatl[i]),
			dbat_is_valid(r.dbatu[i], r.dbatl[i]) ? "yes" : "no");
	}
}

void MemInterface::dumpmem() const {
	dumpToFile(pmem(), MAIN_MEMORY_SIZE, "mainmem.bin");
	if(g::cache_enabled)
		dumpToFile(pcmem(), CACHE_SIZE, "cachemem.bin");
}

void MemInterface::read(DWORD address, DWORD size, void *dest) {
	TRANSLATE_D_READ(address);
	MDEGUB(p_memdegub, "read[%08X] 0x%X bytes ", address, size);
	memcpy(dest, getp_physical(PHYSICALIZE(address), size), size);
}
void MemInterface::write(DWORD address, DWORD size, const void *src) {
	TRANSLATE_D_WRITE(address);
	MDEGUB(p_memdegub, "write[%08X] 0x%X bytes ", address, size);
	memcpy(getp_physical(PHYSICALIZE(address), size), src, size);
}

void MemInterface::read_cached(DWORD address, DWORD size, void *dest) {
	TRANSLATE_D_READ(address);
	MDEGUB(p_memdegub, "read[%08X] 0x%X bytes ", address, size);
	if(address >= CACHE_BASE && address < (CACHE_BASE + CACHE_SIZE)) {
		if((address + size) < (CACHE_BASE + CACHE_SIZE))
			memcpy(dest, getphc(address), size);
		else {
			DWORD cs = (CACHE_BASE + CACHE_SIZE) - address;
			memcpy(dest, getphc(address), cs);
			memcpy((BYTE*)dest + cs, getp_physical(CACHE_SIZE, size - cs),
				size - cs);
		}
	} else
		memcpy(dest, getp_physical(PHYSICALIZE(address), size), size);
}
void MemInterface::write_cached(DWORD address, DWORD size, const void *src) {
	TRANSLATE_D_WRITE(address);
	MDEGUB(p_memdegub, "write[%08X] 0x%X bytes ", address, size);
	if(address >= CACHE_BASE && address < CACHE_BASE + CACHE_SIZE) {
		if((address + size) < CACHE_BASE + CACHE_SIZE)
			memcpy(getphc(address), src, size);
		else {
			DWORD cs = (CACHE_BASE + CACHE_SIZE) - address;
			memcpy(getphc(address), src, cs);
			memcpy(getp_physical(CACHE_SIZE, size - cs), (BYTE*)src + cs,
				size - cs);
		}
	} else
		memcpy(getp_physical(PHYSICALIZE(address), size), src, size);
}

void MemInterface::read_physical(DWORD address, DWORD size, void *dest) {
	memcpy(dest, getp_physical(address, size), size);
}
void MemInterface::write_physical(DWORD address, DWORD size, const void *src) {
	memcpy(getp_physical(address, size), src, size);
}

BYTE *MemInterface::getp_physical(DWORD address, DWORD size) {
	if(address >= MAIN_MEMORY_SIZE || address + size > MAIN_MEMORY_SIZE) {
		if(address >= LOCKED_CACHE_BASE && address + size <= LOCKED_CACHE_END)
			return &locked_cache[address - LOCKED_CACHE_BASE];
		if(address >= EFB_PBASE && address + size <= (EFB_PBASE + EFB_SIZE))
			return &efb[address - EFB_PBASE];
		DEGUB("PAouB: 0x%08X + 0x%08X\n", address, size);
		throw generic_fatal_exception("Physical access out of bounds!");
	}
	return &mainmem[address];
}

void *MemInterface::getphc(DWORD address) {
	if(g::cache_enabled && (address >= CACHE_BASE && address < CACHE_BASE + CACHE_SIZE))
		return &cachemem[address - CACHE_BASE];
	else
		return getph(address);
}
void *MemInterface::getph(DWORD address) {
	if(PHYSICALIZE(address) >= MAIN_MEMORY_SIZE - 8) {
		if(IS_LOCKED_CACHE_ADDRESS(address))
			return &locked_cache[address - LOCKED_CACHE_BASE];
		if(IS_EFB_ADDRESS(address))
			return &efb[address - EFB_BASE];
		DEGUB("EA: 0x%08X\n", address);
		//NONPUBLIC_FORCED_CRASH;
		throw generic_fatal_exception("Invalid memory address!");
	}
	return &mainmem[PHYSICALIZE(address)];
}
void *MemInterface::getpph(DWORD paddress) {
	if(paddress >= MAIN_MEMORY_SIZE - 8) {
		if(IS_LOCKED_CACHE_ADDRESS(paddress))
			return &locked_cache[paddress - LOCKED_CACHE_BASE];
		if(IS_EFB_PADDRESS(paddress))
			return &efb[paddress - EFB_PBASE];
		DEGUB("PA: 0x%08X\n", paddress);
		//NONPUBLIC_FORCED_CRASH;
		throw generic_fatal_exception("Invalid memory address!");
	}
	return &mainmem[paddress];
}
